package com.uc.crashsdk.demo;

import android.support.v7.app.AppCompatActivity;
import android.os.Bundle;
import android.util.Log;
import android.view.View;

import com.uc.crashsdk.JNIBridge;

import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

public class MainActivity extends AppCompatActivity implements View.OnClickListener {

    final private static String TAG = "demo";

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

        // register onClick() event
        int[] btnIds = new int[] {
                R.id.java_null_ptr, R.id.java_oom, R.id.java_fd_leak,
                R.id.native_crash, R.id.native_heap_corruption, R.id.native_fd_leak,
                R.id.native_abort, R.id.native_stack_overflow, R.id.native_oom,
                R.id.unexp_kill_process, R.id.unexp_exit, R.id.unexp_anr,
        };
        for (int btnId : btnIds) {
            findViewById(btnId).setOnClickListener(this);
        }
    }

    @Override
    public void onClick(View view) {
        doCrash(view.getId());
    }

    public void doCrash(int id) {
        switch (id) {
            case R.id.java_null_ptr:
                crashInJava();
                break;

            case R.id.java_oom:
                javaOOMCrash();
                break;

            case R.id.java_fd_leak:
                useOutAllFileHandles();
                crashInJava();
                break;

            case R.id.native_crash:
                nativeCrash(0);
                break;

            case R.id.native_heap_corruption:
                nativeCrash(1);
                break;

            case R.id.native_fd_leak:
                useOutAllFileHandles();
                nativeCrash(0);
                break;

            case R.id.native_abort:
                nativeCrash(2);
                break;

            case R.id.native_stack_overflow:
                nativeCrash(3);
                break;

            case R.id.native_oom:
                nativeCrash(5);
                break;

            case R.id.unexp_kill_process:
                android.os.Process.killProcess(android.os.Process.myPid());
                break;

            case R.id.unexp_exit:
                nativeCrash(4);
                break;

            case R.id.unexp_anr:
                while (true) {
                    try {
                        Thread.sleep(1);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                }

            default:
                break;
        }
    }

    private List<FileInputStream> mFiles = new ArrayList<FileInputStream>();

    private void useOutAllFileHandles() {
        for (int i = 0; i < 2048; ++i) {
            try {
                mFiles.add(new FileInputStream("/dev/null"));
            } catch (Exception e) {
                e.printStackTrace();
                Log.v(TAG, "!!!!! too many open files!");
                break;
            }
        }
    }

    private void nativeCrash(int type) {
        JNIBridge.nativeCrash(type, 0);
    }

    private void crashInJava() {
        String nullStr = "1";
        if (nullStr.equals("1")) {
            nullStr = null;
        }
        nullStr.equals("");
    }

    private List<byte[]> mMems = new ArrayList<byte[]>(4096);

    private void javaOOMCrash() {
        final int kInitSize = 10 * 1024 * 1024;
        final int kMinSize = 1024;

        int size = kInitSize;
        int totalAllocSize = 0;
        while (size > 0) {
            try {
                byte[] mem = new byte[size];
                for (int i = 1; i < size; i += 4096) {
                    mem[i] = (byte) i;
                }
                mMems.add(mem);
                totalAllocSize += size;
            } catch (OutOfMemoryError t) {
                if (size < kMinSize) {
                    Log.w(TAG, String.format(Locale.US,
                            "=Total %d bytes", totalAllocSize));
                    throw t;
                }
                size /= 2;
            }
        }
        // Crash what ever
        byte[] mem = new byte[kInitSize];
        mMems.add(mem);
    }

}
