package com.uc.crashsdk.tests.standard;

import java.io.File;
import java.util.ArrayList;

import android.content.Context;
import android.telephony.TelephonyManager;
import android.util.Log;

import com.uc.crashsdk.export.CrashApi;
import com.uc.crashsdk.export.CustomInfo;
import com.uc.crashsdk.export.CustomLogInfo;
import com.uc.crashsdk.export.ExitType;
import com.uc.crashsdk.export.ICrashClient;
import com.uc.crashsdk.export.LogType;
import com.uc.crashsdk.export.VersionInfo;

/**
 *
 * This sample is a standard use with crashsdk, which includes java, native,
 * unexp logs generating, crash stats, custom log generating, and add custom
 * informations into logs before crash or while crashing.
 *
 */
public class CrashWrapper {

    protected static final String TAG = "crashsdk";

    public static boolean IS_DEBUG_VERSION = true;
    public static boolean IS_PUBLIC_RELEASE_VERSION = false;

    final public static boolean ENANBLE_JAVA_LOG   = true;
    final public static boolean ENABLE_NATIVE_LOG  = true;
    final public static boolean ENABLE_UNEXP_LOG   = true;

    private static CrashApi sCrashApi = null;
    private static Context  sContext  = null;

    public static void initialize(Context context) {
        Log.i(TAG, "CrashWrapper.initialize");
        sContext = context;
        final String url = "http://up4.ucweb.com:8012/upload";
        sCrashApi = CrashApi.createInstance(context, getCustomInfo(),
                getVersionInfo(), new CrashClientImpl(), url,
                ENANBLE_JAVA_LOG, ENABLE_NATIVE_LOG, ENABLE_UNEXP_LOG);
    }

    public static void setNewInstall() {
        sCrashApi.setNewInstall();
    }

    public static void uploadCrashLog() {
        Log.d(TAG, "uploadCrashLogs, server url: "
                + sCrashApi.getCrashLogUploadUrl());
        sCrashApi.uploadCrashLogs();
    }

    public static void setForeground(boolean isForeground) {
        sCrashApi.setForeground(isForeground);
    }

    public static boolean onExit() {
        sCrashApi.onExit();
        return true;
    }

    public static void registerThread(String name) {
        sCrashApi.registerThread(LogType.JAVA | LogType.NATIVE, name);
    }

    public static void prepareCrashInfo(String soPath, boolean isNewInstall) {
        @SuppressWarnings("unused")
        final boolean needLoadCrashsdkSo = ENABLE_NATIVE_LOG || ENABLE_UNEXP_LOG;

        if (needLoadCrashsdkSo) {
            // NOTICE: some service process may not need native log,
            // and no need load libcrashsdk.so.
            loadCrashsdkSo(soPath);

            if (ENABLE_NATIVE_LOG) {
                Log.i(TAG, "crashSoLoaded");
                sCrashApi.crashSoLoaded();

                // Add the thread, so that it will dump the thread stack into
                // jni crash logs.
                // Pass null for the |name|, it will auto get the thread name.
                registerThread(null);
            }
        }

        // IMPORTANT: as early as possible
        sCrashApi.setMainProcess();
        sCrashApi.setForeground(true);

        if (isNewInstall) {
            Log.i(TAG, "setNewInstall");
            setNewInstall();
        }

        Log.i(TAG, "lastExitType: " + exitTypeToString(sCrashApi.getLastExitType()));

        // TEST: add header info
        sCrashApi.addHeaderInfo("imei", getIMEI());
        sCrashApi.addHeaderInfo("header key", "header value");

        final int LOG_NATIVE_JAVA   = LogType.NATIVE | LogType.JAVA;

        // TEST: add callback info.
        sCrashApi.registerInfoCallback(CallbackTest, LOG_NATIVE_JAVA);

        // update the version info after incremental upgraded.
        //updateVersionInfoTest();
    }

    private static String exitTypeToString(int exitType) {
        switch (exitType) {
        case ExitType.NORMAL_EXIT:
            return "NORMAL_EXIT";

        case ExitType.FOREGROUND_CAUGHT_CRASH:
            return "FOREGROUND_CAUGHT_CRASH";

        case ExitType.BACKGROUND_CAUGHT_CRASH:
            return "BACKGROUND_CAUGHT_CRASH";

        case ExitType.UNKNOWN_GROUND_CAUGHT_CRASH:
            return "UNKNOWN_GROUND_CAUGHT_CRASH";

        case ExitType.FOREGROUND_UNCAUGHT_CRASH:
            return "FOREGROUND_UNCAUGHT_CRASH";

        case ExitType.BACKGROUND_UNCAUGHT_CRASH:
            return "BACKGROUND_UNCAUGHT_CRASH";

        default:
            return "UNKNOWN: " + exitType;
        }
    }

    public static String getIMEI() {
        String imei = null;
        try {
            TelephonyManager telephonyMgr = (TelephonyManager) sContext
                    .getSystemService(Context.TELEPHONY_SERVICE);
            imei = telephonyMgr.getDeviceId();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return imei;
    }

    public static void reportCrashStats() {
        sCrashApi.reportCrashStats(false);
    }

    public static void resetCrashStats() {
        sCrashApi.resetCrashStats(false);
    }

    public static boolean generateCustomLog() {
        StringBuffer buffer = new StringBuffer();
        buffer.append("Any strings as you want write into the log from java.");
        CustomLogInfo info = new CustomLogInfo(buffer, "mytype");

        info.mAddThreadsDump = true;

        info.mDumpTids = new ArrayList<Integer>();
        info.mDumpTids.add(android.os.Process.myPid());
        info.mDumpTids.add(android.os.Process.myTid());

        info.mCallbacks = new ArrayList<String>();
        info.mCallbacks.add(CallbackTest);

        return sCrashApi.generateCustomLog(info);
    }

    //////////////////////////////////////////////////////////////////////////

    private static void loadCrashsdkSo(String path) {
        File file = new File(path, "libcrashsdk.so");
        if (file.exists()) {
            System.load(file.getPath());
        } else {
            System.loadLibrary("crashsdk");
        }
    }

    @SuppressWarnings("unused")
    private static void updateVersionInfoTest() {
        VersionInfo info = new VersionInfo();
        info.mVersion    = "1.1.1.1";
        info.mSubVersion = "release";
        info.mBuildSeq   = "151122000000";
        sCrashApi.updateVersionInfo(info);
    }

    private static CustomInfo getCustomInfo() {
        CustomInfo info = new CustomInfo();

        // The name of your APP. Do not contain char '_'.
        info.mCrashLogPrefix        = "CrashSDK";

        // true: generate unexp log only if it's ANR or other known reason
        // false: generate unexp log for all uncaught crash.
        info.mUnexpOnlyKnownReason  = IS_PUBLIC_RELEASE_VERSION;

        // Need declare com.uc.crashsdk.CrashLogFilesUploader in file
        // AndroidManifest.xml
        info.mUploadUcebuCrashLog   = true;

        // Zip the logs after generated.
        info.mZipLog                = !IS_DEBUG_VERSION;

        // Copy plain text crash logs (no zip) into /sdcard/XXX/ or 
        //  CustomInfo.mLogsBackupPathName if it has been set.
        // (XXX is CustomInfo.mCrashLogsFolderName)
        info.mBackupLogs            = IS_DEBUG_VERSION;

        // true: print crash informations into logcat while generating logs.
        info.mDebugCrashSDK         = IS_DEBUG_VERSION;

        // true: enable crash stat.
        info.mEnableStatReport      = true;

        return info;
    }

    private static VersionInfo getVersionInfo() {
        VersionInfo info = new VersionInfo();

        info.mVersion    = "1.0.0.1";
        info.mSubVersion = "beta";
        info.mBuildSeq   = "150102001122"; // suggested as "yymmddHHmmss"

        return info;
    }

    public static final String CallbackTest = "vm info:";

    private static class CrashClientImpl implements ICrashClient {

        @Override
        public void onLogGenerated(File file, String logType) {
            Log.i(TAG, "onLogGenerated: " + file + ", type: " + logType
                    + ", isCrash: " + isCrashType(logType) + ", exist: "
                    + file.exists() + ", len: " + file.length());
        }

        private boolean isCrashType(String logType) {
            if (logType == null || "".equals(logType)) {
                return false;
            }
            return LogType.JAVA_TYPE.equals(logType)
                    || LogType.NATIVE_TYPE.equals(logType);
        }

        @Override
        public void onCrashRestarting(boolean isJava) {
            Log.i(TAG, "onCrashRestarting, isJava: " + isJava);
        }

        @Override
        public boolean onAddCrashStats(String processName, int key, int count) {
            Log.i(TAG, "onAddCrashStats, processName: " + processName
                    + ", key: " + key + ", count: " + count);
            return false;
        }

        @Override
        public String onGetCallbackInfo(String category) {
            Log.i(TAG, "onGetCallbackInfo, description: " + category);

            // The return strings will be wrote into crash logs.
            if (CallbackTest.equals(category)) {
                return "MaxMemory: " + Runtime.getRuntime().maxMemory()
                        + "\nTotalMemory: " + Runtime.getRuntime().totalMemory()
                        + "\nFreeMemory: " + Runtime.getRuntime().freeMemory();

            }

            return "";
        }

        @Override
        public File onBeforeUploadLog(File file) {
            return file;
        }

    }
}
