package com.uc.crashsdk.tests.native_api;

import java.io.File;

import android.content.Context;
import android.util.Log;

import com.uc.crashsdk.export.CrashApi;
import com.uc.crashsdk.export.CustomInfo;
import com.uc.crashsdk.export.LogType;
import com.uc.crashsdk.export.VersionInfo;

/**
 *
 * This sample shows how to use crashsdk native APIs.
 * It loads libnativetest.so, and add header info, add dump file, add callback
 * info into logs in the native. It also shows how to generate a custom log
 * in the native.
 *
 */
public class CrashWrapper {

    protected static final String TAG = "crashsdk";

    final public static boolean ENANBLE_JAVA_LOG   = true;
    final public static boolean ENABLE_NATIVE_LOG  = true;
    final public static boolean ENABLE_UNEXP_LOG   = false;

    private static final boolean IS_PUBLIC_RELEASE_VERSION = false;
    private static final boolean IS_DEBUG_VERSION = true;

    private static CrashApi sCrashApi = null;

    public static void initialize(Context context) {
        Log.i(TAG, "CrashWrapper_Bundle.initialize");
        final String url = "http://up4.ucweb.com:8012/upload";
        sCrashApi = CrashApi.createInstance(context, getCustomInfo(),
                getVersionInfo(), url);
    }

    public static void uploadCrashLog() {
        Log.d(TAG, "uploadCrashLogs, server url: "
                + sCrashApi.getCrashLogUploadUrl());
        sCrashApi.uploadCrashLogs();
    }

    public static void prepareCrashInfo(String soPath, boolean isNewInstall) {
        // 1. load libcrashsdk.so
        loadCrashsdkSo(soPath);
        sCrashApi.crashSoLoaded();
        Log.v(TAG, "crashSoLoaded");

        // 2. load libnativetest.so
        // libnativetest.so depends on libcrashsdk.so, so it must load after
        // libcrashsdk.so
        loadNativeTestSo(soPath);
        Log.v(TAG, "libnativetest.so loaded");

        // 3. after loaded libnativetest.so, call to native, test native APIs.
        nativePrepareCrashInfos();

        registerThread(null);
    }

    public static boolean generateCustomLog() {
        // Generate a custom log in native code.
        // We can use CrashApi.generateCustomLog to generate a custom log
        // in java, do not need use native API, here is just test native API.
        return nativeGenerateCustomLog();
    }

    public static void registerThread(String name) {
        sCrashApi.registerThread(LogType.NATIVE | LogType.JAVA, name);
    }

    public static void setNewInstall() { }
    public static void setForeground(boolean isForeground) { }
    public static boolean onExit() { return false; }
    public static void reportCrashStats() { }
    public static void resetCrashStats() { }

    //////////////////////////////////////////////////////////////////////////

    private static void loadCrashsdkSo(String path) {
        File file = new File(path, "libcrashsdk.so");
        if (file.exists()) {
            System.load(file.getPath());
        } else {
            System.loadLibrary("crashsdk");
        }
    }

    private static void loadNativeTestSo(String path) {
        File file = new File(path, "libnativetest.so");
        if (file.exists()) {
            System.load(file.getPath());
        } else {
            System.loadLibrary("nativetest");
        }
    }

    private static CustomInfo getCustomInfo() {
        CustomInfo info = new CustomInfo();

        info.mCrashLogPrefix        = "CrashSDK";

        // true: generate unexp log only if it's ANR or other known reason
        // false: generate unexp log for all uncaught crash.
        info.mUnexpOnlyKnownReason  = IS_PUBLIC_RELEASE_VERSION;

        // Need declare com.uc.crashsdk.CrashLogFilesUploader in file
        // AndroidManifest.xml
        info.mUploadUcebuCrashLog   = true;

        info.mZipLog                = !IS_DEBUG_VERSION;
        info.mBackupLogs            = IS_DEBUG_VERSION;
        info.mDebugCrashSDK         = IS_DEBUG_VERSION;

        return info;
    }

    private static VersionInfo getVersionInfo() {
        VersionInfo info = new VersionInfo();

        info.mVersion    = "1.0.0.1";
        info.mSubVersion = "beta";
        info.mBuildSeq   = "150102001122"; // suggested as "yymmddHHmmss"

        return info;
    }

    private static native void nativePrepareCrashInfos();
    private static native boolean nativeGenerateCustomLog();

}
