package com.uc.crashsdk.tests;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.util.Log;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewParent;
import android.view.Window;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemClickListener;
import android.widget.ArrayAdapter;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.RadioButton;
import android.widget.RadioGroup;
import android.widget.Toast;

import com.uc.crashsdk.JNIBridge;

public class TestCrashSDKActivity extends Activity {

    final private static String ACTION_JAVA_CRASH        = "java crash";
    final private static String ACTION_JAVA_OOM_CRASH    = "java oom crash";
    final private static String ACTION_JAVA_EMFILE_CRASH = "java too many open files crash";

    final private static String ACTION_KILL              = "kill in java (unexp)";

    final private static String ACTION_JNI_CRASH                = "native crash";
    final private static String ACTION_JNI_HEAP_CRASH           = "native heap corruption";
    final private static String ACTION_JNI_EMFILE_CRASH         = "native too many open files crash";
    final private static String ACTION_JNI_ABORT_CRASH          = "native abort (test for android 5.x)";
    final private static String ACTION_JNI_STACKOVERFLOW_CRASH  = "native stack overflow";
    final private static String ACTION_JNI_EXIT                 = "native exit (unexp)";

    final private static String ACTION_JAVA_BG_CRASH    = "java bg crash";
    final private static String ACTION_JNI_BG_CRASH     = "native bg crash";

    final private static String ACTION_GENERATE_CUSTOM_LOG = "generate custom log";
    final private static String ACTION_LOCAL_UNWIND  = "backtrace main thread";

    final private static String DISALBED = " (disabled)";

    protected static final String TAG = "crashsdk";

    private static List<String> TEST_ITEMS = new ArrayList<String>();

    private static Handler         mUIHandler = null;

    private static HandlerThread   mMyThread = null;
    private static Handler         mMyHandler = null;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        if (mUIHandler == null) {
            mUIHandler = new Handler(Looper.getMainLooper());
        }

        if (mMyThread == null) {
            CrashWrapperSelector.prepareCrashInfo(getApplicationInfo().dataDir + "/lib",
                    checkNewInstall());

            mMyThread = new HandlerThread("MyThread");
            mMyThread.start();
            mMyHandler = new Handler(mMyThread.getLooper());

            mMyHandler.post(new Runnable() {

                @Override
                public void run() {
                    // Call registerThread, so that the crash log will record
                    // the stack of current thread although it's not the crash
                    // thread.
                    // Another benefit of call registerThread is to avoid
                    // generate log failed with native stack overflow.
                    //
                    // Register some important thread is needed, but there is
                    // no need to register each thread.
                    CrashWrapperSelector.registerThread(null);
                }

            });

            uploadCrashLogs();
            reportCrashStats();
        }

        createItems();
    }

    private static LinearLayout mContainer = null;
    private static ListView mUrlListView = null;
    private static RadioGroup mCrashThreadRadio = null;


    private void createItems() {
        if (mContainer == null) {
            boolean javaEnabled = CrashWrapperSelector.ENANBLE_JAVA_LOG;
            String java = javaEnabled ? "" : DISALBED;
            TEST_ITEMS.add(ACTION_JAVA_CRASH + java);
            TEST_ITEMS.add(ACTION_JAVA_OOM_CRASH + java);
            TEST_ITEMS.add(ACTION_JAVA_EMFILE_CRASH + java);
            TEST_ITEMS.add(ACTION_JAVA_BG_CRASH + java);

            boolean jniEnabled = CrashWrapperSelector.ENABLE_NATIVE_LOG;
            String jni = jniEnabled ? "" : DISALBED;

            boolean unexpEnabled = CrashWrapperSelector.ENABLE_UNEXP_LOG;
            String unexp = unexpEnabled ? "" : DISALBED;

            TEST_ITEMS.add(ACTION_KILL + unexp);

            TEST_ITEMS.add(ACTION_JNI_CRASH + jni);
            TEST_ITEMS.add(ACTION_JNI_HEAP_CRASH + jni);
            TEST_ITEMS.add(ACTION_JNI_EMFILE_CRASH + jni);
            TEST_ITEMS.add(ACTION_JNI_ABORT_CRASH + jni);
            TEST_ITEMS.add(ACTION_JNI_STACKOVERFLOW_CRASH + jni);
            TEST_ITEMS.add(ACTION_JNI_EXIT + jni);
            TEST_ITEMS.add(ACTION_JNI_BG_CRASH + jni);

            TEST_ITEMS.add(ACTION_GENERATE_CUSTOM_LOG);
            TEST_ITEMS.add(ACTION_LOCAL_UNWIND);

            requestWindowFeature(Window.FEATURE_PROGRESS);

            mContainer = new LinearLayout(this);
            mContainer.setOrientation(LinearLayout.VERTICAL);

            mCrashThreadRadio = new RadioGroup(this);
            RadioButton tmp = new RadioButton(this);
            tmp.setText("In non-main thread");
            mCrashThreadRadio.addView(tmp);
            tmp.setChecked(true);

            tmp = new RadioButton(this);
            tmp.setText("In main thread");
            mCrashThreadRadio.addView(tmp);

            tmp = new RadioButton(this);
            tmp.setText("In pure native thread (native crash)");
            mCrashThreadRadio.addView(tmp);

            mContainer.addView(mCrashThreadRadio);

            mUrlListView = new ListView(this);
            mUrlListView.setAdapter(new ArrayAdapter<String>(this,
                    android.R.layout.simple_list_item_1, TEST_ITEMS));
            mUrlListView.setTextFilterEnabled(true);

            mContainer.addView(mUrlListView);

            registerClickEvents();
        }

        ViewParent parent = mContainer.getParent();
        if (parent != null && parent instanceof ViewGroup) {
            ViewGroup parentView = (ViewGroup) parent;
            parentView.removeView(mContainer);
        }
        setContentView(mContainer);
    }

    private boolean crashInMainThread() {
        return mCrashThreadRadio != null
                && mCrashThreadRadio.getCheckedRadioButtonId()
                    == mCrashThreadRadio.getChildAt(1).getId();
    }

    private boolean crashInPureNativeThread() {
        return mCrashThreadRadio != null
                && mCrashThreadRadio.getCheckedRadioButtonId()
                    == mCrashThreadRadio.getChildAt(2).getId();
    }

    private void nativeCrash(int type) {
        JNIBridge.nativeCrash(type, crashInPureNativeThread() ? 1 : 0);
    }

    private void doCrash(String action, Handler handler) {
        if (action.indexOf(ACTION_KILL) >= 0) {
            android.os.Process.killProcess(android.os.Process.myPid());
        } else if (action.indexOf(ACTION_JNI_CRASH) >= 0) {
            nativeCrash(0);
        } else if (action.indexOf(ACTION_JNI_HEAP_CRASH) >= 0) {
            nativeCrash(1);
        } else if (action.indexOf(ACTION_JNI_EMFILE_CRASH) >= 0) {
            useOutAllFileHandles();
            nativeCrash(0);
        } else if (action.indexOf(ACTION_JNI_ABORT_CRASH) >= 0) {
            nativeCrash(2);
        } else if (action.indexOf(ACTION_JNI_STACKOVERFLOW_CRASH) >= 0) {
            nativeCrash(3);
        } else if (action.indexOf(ACTION_JNI_EXIT) >= 0) {
            nativeCrash(4);
        } else if (action.indexOf(ACTION_JNI_BG_CRASH) >= 0) {
            moveTaskToBack(true);

            handler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    nativeCrash(0);
                }
            }, 1000);
        } else if (action.indexOf(ACTION_JAVA_CRASH) >= 0) {
            crashInJava();
        } else if (action.indexOf(ACTION_JAVA_OOM_CRASH) >= 0) {
            javaOOMCrash();
        } else if (action.indexOf(ACTION_JAVA_EMFILE_CRASH) >= 0) {
            useOutAllFileHandles();
            crashInJava();
        } else if (action.indexOf(ACTION_JAVA_BG_CRASH) >= 0) {
            moveTaskToBack(true);

            handler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    crashInJava();
                }
            }, 1000);
        } else if (action.indexOf(ACTION_GENERATE_CUSTOM_LOG) >= 0) {
            boolean res = CrashWrapperSelector.generateCustomLog();
            final String str = "generateCustomLog "
                    + (res ? "succeed" : "failed. It may reach"
                            + " CustomInfo.mMaxCustomLogCountPerTypePerDay");
            Toast.makeText(this, str, Toast.LENGTH_LONG).show();
            Log.d(TAG, action);
        } else if (action.indexOf(ACTION_LOCAL_UNWIND) >= 0) {
            nativeCrash(100);
        }
    }

    private void registerClickEvents() {
        final Context main = this;
        mUrlListView.setOnItemClickListener(new OnItemClickListener() {

            @Override
            public void onItemClick(AdapterView<?> parent, View view,
                    int position, long id) {

                final String action = TEST_ITEMS.get(position);
                Toast.makeText(main, action, Toast.LENGTH_LONG).show();
                Log.d(TAG, action);

                if (crashInMainThread()) {
                    doCrash(action, mUIHandler);
                } else {
                    mMyHandler.post(new Runnable() {

                        @Override
                        public void run() {
                            doCrash(action, mMyHandler);
                        }
                    });
                }
            }
        });
    }

    private boolean checkNewInstall() {
        boolean isNewInstall = false;

        String newInstallFile = getApplicationInfo().dataDir
                + "/" + TestReceiver.NEW_INSTALL_FILE;
        File file = new File(newInstallFile);
        if (file.exists()) {
            file.delete();
            isNewInstall = true;
        }

        String startedFile = getApplicationInfo().dataDir
                + "/" + TestReceiver.STARTED_FILE;
        File f = new File(startedFile);
        if (!f.exists()) {
            isNewInstall = true;
            try {
                f.createNewFile();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return isNewInstall;
    }

    private final List<FileInputStream> mFiles = new ArrayList<FileInputStream>();

    void useOutAllFileHandles() {
        for (int i = 0; i < 1024; ++i) {
            try {
                FileInputStream input = new FileInputStream("/dev/null");
                mFiles.add(input);
            } catch (Exception e) {
                e.printStackTrace();
                Log.v(TAG, "!!!!! too many open files!");
                break;
            }
        }
    }

    void crashInJava() {
        String nullStr = "1";
        if (nullStr.equals("1")) {
            nullStr = null;
        }
        nullStr.equals("");
    }

    void javaOOMCrash() {
        throw new OutOfMemoryError("test out of memory");
    }

    private void uploadCrashLogs() {
        // post delayed, so it will not effect the APP startup speed.
        mMyHandler.postDelayed(new Runnable() {

            @Override
            public void run() {
                // If has first net connection, call uploadCrashLogs
                CrashWrapperSelector.uploadCrashLog();
            }

        }, 10000);
    }

    public void reportCrashStats() {
        mMyHandler.postDelayed(new Runnable() {

            @Override
            public void run() {
                // report the crash stats, callback with ICrashCallback.addCrashStats
                CrashWrapperSelector.reportCrashStats();

                // clear manual, or return true in ICrashCallback.addCrashStats
                // will reset the crash too.
                //CrashWrapperSelector.resetCrashStats();
            }

        }, 5000);
    }

    @Override
    protected void onPause() {
        super.onPause();
        // call at ground changed.
        CrashWrapperSelector.setForeground(false);
        Log.v(TAG, "setForeground(false), onPause");
    }

    @Override
    protected void onStop() {
        super.onStop();
        CrashWrapperSelector.setForeground(false);
        Log.v(TAG, "setForeground(false), onStop");
    }

    @Override
    protected void onResume() {
        super.onResume();
        CrashWrapperSelector.setForeground(true);
        Log.v(TAG, "setForeground(true), onResume");
    }

    @Override
    protected void onStart() {
        super.onStart();
        CrashWrapperSelector.setForeground(true);
        Log.v(TAG, "setForeground(true), onStart");
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        CrashWrapperSelector.onExit();
        Log.v(TAG, "onDestroy, CrashWrapper.onExit");
    }

    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event) {
        if(keyCode == KeyEvent.KEYCODE_BACK) {
            // call it before normal exit if needed.
            if (CrashWrapperSelector.onExit()) {
                android.os.Process.killProcess(android.os.Process.myPid());
            }
            Log.v(TAG, "onKeyDown, CrashWrapper.onExit");
        }
        return super.onKeyDown(keyCode, event);
    }

}
