#include <jni.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/mman.h>
#include <android/log.h>

#include "crashsdk.h"


#define TEST_LOG(...)  __android_log_print(ANDROID_LOG_INFO, "crashsdk", __VA_ARGS__)


const char* testInfoCallbackFunc(const char* category, LogType type, long* dataSize)
{
    const int kBufferSize = getpagesize(); // 4KB

    // It's better do not use libc malloc or free here, which may crashes
    // with the libc heap corruption. It better use mmap system call to
    // allocate a buffer.
    //char* buffer = (char*)malloc(kBufferSize);
    char* buffer = (char*)mmap(NULL, kBufferSize, PROT_READ | PROT_WRITE,
                MAP_PRIVATE | MAP_ANONYMOUS, -1, 0);

    int len = snprintf(buffer, kBufferSize,
            "enter native callback function.\n"
            "category: '%s', type: %d.\n"
            "you can add any content as you want.",
            category, type);
    *dataSize = len;

    // REMIND: the buffer can not be free-d!
    return buffer;
}


void PrepareCrashInfos(JNIEnv* env, jobject obj)
{
    TEST_LOG("PrepareCrashInfos");

    // 1. TEST FOR add header info
    crashsdk_addHeaderInfo("native HeaderInfo key", "any string as value");

    // 2. TEST FOR add cached info
    const char* cachedBufferKey = "test native cached buffer:";
    crashsdk_createCachedInfo(cachedBufferKey, 10, LogTypeNative | LogTypeJava);
    char buffer[256] = { 0 };
    for (int i = 0; i < 15; ++i) {
       sprintf(buffer, "native data line %d", i);
       crashsdk_addCachedInfo(cachedBufferKey, buffer, strlen(buffer));
    }

    // 3. TEST FOR add callback
    crashsdk_registerInfoCallback("test native callback:",
            LogTypeNative | LogTypeJava, testInfoCallbackFunc);

    // 4. TEST FOR add dump file
    DumpFileInfo info;
    memset(&info, 0, sizeof(info));
    char tmp[32] = { 0 };
    snprintf(tmp, 32, "/proc/%d/stack", getpid());

    info.infoSize           = sizeof(info);
    info.category           = "linux kernel stack:";
    info.fileTobeDump       = tmp;
    info.logType            = LogTypeNative | LogTypeJava | LogTypeUnexp;
    info.writeCategory      = true;
    info.deleteAfterDump    = false;
    crashsdk_addDumpFile(&info);
}

jboolean GenerateCustomLog(JNIEnv* env, jobject obj)
{
    // TEST FOR generate custom log
    CustomLogInfo info;
    memset(&info, 0, sizeof(info));

    const char* datas = "Any string that you want write into the log from native";
    info.infoSize   = sizeof(info);
    info.datas      = datas;
    info.dataSize   = strlen(datas);
    info.logType    = "mytype";

    info.addHeader  = true;
    info.addFooter  = true;
    info.addLogcat  = true;
    info.uploadNow  = false;

    info.addThreadsDump = true;

    info.dumpFiles   = "linux kernel stack:";
    info.callbacks   = "test native callback:";
    info.cachedInfos = "test native cached buffer:";

    int tids[2] = { getpid(), gettid() };
    info.dumpTids = tids;
    info.dumpTidCount = 2;

    return crashsdk_generateCustomLog(&info);
}

static const JNINativeMethod sMethods[] = {
        { "nativePrepareCrashInfos", "()V", (void*) PrepareCrashInfos },
        { "nativeGenerateCustomLog", "()Z", (void*) GenerateCustomLog },
};

bool registeNativeMethods(JNIEnv* env)
{
    if (!env) {
        return false;
    }

    jclass cls = env->FindClass("com/uc/crashsdk/tests/native_api/CrashWrapper");
    if (!cls) {
        return false;
    }

    const int count = sizeof(sMethods) / sizeof(sMethods[0]);
    env->RegisterNatives(cls, sMethods, count);
    env->DeleteLocalRef(cls);

    return true;
}

JNIEXPORT jint JNI_OnLoad(JavaVM* vm, void* reserved)
{
    JNIEnv* env = 0;
    if (vm->GetEnv((void**) &env, JNI_VERSION_1_6) != JNI_OK) {
        return -1;
    }

    if (!registeNativeMethods(env)) {
        return -2;
    }

    return JNI_VERSION_1_6;
}

JNIEXPORT void JNI_OnUnload(JavaVM* vm, void* reserved)
{
}
