// by shijiang.ysj@alibaba-inc.com

#ifndef _CRASH_SDK_H_
#define _CRASH_SDK_H_

#ifdef __cplusplus
extern "C" {
#endif

typedef int     CrashSDK_BOOL;

typedef enum {
    LogTypeNative  = 0x001,
    LogTypeJava    = 0x010,
    LogTypeUnexp   = 0x100
} LogType;

struct DumpFileInfo {
    long infoSize;

    const char* category;
    const char* fileTobeDump;
    int         logType;

    CrashSDK_BOOL  writeCategory;
    CrashSDK_BOOL  deleteAfterDump;
    CrashSDK_BOOL  isEncrypted;     // UCM private
};

struct CustomLogInfo {
    long infoSize;

    const char* datas;
    long        dataSize;
    const char* logType;

    CrashSDK_BOOL  addHeader;
    CrashSDK_BOOL  addFooter;
    CrashSDK_BOOL  addLogcat;
    CrashSDK_BOOL  uploadNow;

    CrashSDK_BOOL  addThreadsDump;

    const char*    dumpFiles;   // Use ';' to separate multiple dump file
    const char*    callbacks;   // Use ';' to separate multiple callback
    const char*    cachedInfos; // Use ';' to separate multiple cached info

    const int*    dumpTids;
    int           dumpTidCount;
};

typedef const char* (*InfoCallbackFunc)(const char* category,
        LogType logType, long* dataSize);


// Add header info into all logs (Native, Java, Unexp, and custom logs)
//
// itemName can not be NULL or "".
// itemValue can not be NULL.
// return succeed LogType(s).
int crashsdk_addHeaderInfo(const char* itemName, const char* itemValue);

// Register current thread to |logType|. Java and Native log will print the
// call stack of the registered threads into crash logs while crashing.
//
// name can be NULL or "".
// logType must has one or more flags of LogType.
// return succeed LogType(s).
int crashsdk_registerThread(const char* name, int logType);

// Register a callback |func| to |logType|. CrashSDK will callback with |func|
// while generating crash logs to get some text, and wrote them into logs.
//
// Java can only add 8 callbacks max.
// Native can only add 4 callbacks max.
// Unexp can only add 6 callbacks max.
//
// category can not be NULL or "".
// logType must has one or more flags of LogType.
// func can not be NULL.
// return succeed LogType(s).
int crashsdk_registerInfoCallback(const char* category, int logType,
        InfoCallbackFunc func);

// Add a file into crash logs. CrashSDK will dump the content of the file
//
// Each type of log can only add 10 dump files max.
//
// into logs while generating crash logs.
// info can not be NULL.
// info->infoSize must be the value of sizeof(DumpFileInfo).
// info->category can not be NULL or "".
// info->fileTobeDump can not be NULL or "".
// info->logType must has one or more flags of LogType.
// return succeed LogType(s).
int crashsdk_addDumpFile(const DumpFileInfo* info);

// Generate a custom log. It can use consistent format with Java or
// Native logs.
//
// info can not be NULL.
// info->infoSize must be the value of sizeof(CustomLogInfo).
// info->datas can not be NULL.
// info->dataSize must be > 0.
// info->logType can not contain char '_'.
// return true if generate log succeed, otherwise return false.
bool crashsdk_generateCustomLog(const CustomLogInfo* info);

// Create a cached information item. It will keep the last |capacity|
// lines that added by addCachedInfo.
//
// Each type of log can only add 3 cached info(s) max.
//
// category can not be NULL or "".
// capacity must > 0.
// logType must has one or more flags of LogType.
// return succeed LogType(s).
int crashsdk_createCachedInfo(const char* category, int capacity, int logType);

// Add a cached info into crash logs. If the cache is exceed the size
// that specified by createCachedInfo, it will remove the first info.
//
// category can not be NULL or "", and must has called with createCachedInfo.
// data can not be NULL.
// dataSize, the size of data, must >= 0.
// return succeed LogType(s).
int crashsdk_addCachedInfo(const char* category, const char* data, long dataSize);


#ifdef __cplusplus
}
#endif

#endif // _CRASH_SDK_H_
